package com.junkprojects.comm.rpc;

import java.io.*;

public class RPCOutputStream extends DataOutputStream {

  //Note on the API, the reason I made the methods for the
  //wirting the specific types of messages public instead of
  //private like in RPCInputStream, is I think code using
  //this class will be more readable if the programmer uses
  //the specific call. With RPCInputStream, you may not 
  //be sure of what kind of message is being sent so it seemed
  //better for the stream to figure it out for you and
  //give you an instance of what came across the wire -PICKARD

  private OutputStream _basestream = null;
  RPCXidGen xidgen  = null;

  public RPCOutputStream(OutputStream os) {
    super(os);

    _basestream = os;

     xidgen = RPCXidGen.getInstance();
  }

  public void writeMessage(RPCMessage message) 
         throws RPCIOException,
                IOException
  {

    int messagetype = -1;

    if (message == null) 
      throw new RPCIOException("Can not write a null message");

    messagetype = message.getMessageType();

    if(messagetype == RPCMessage.MSGTYPE_CALL) {
      try {
          writeCallMessage((RPCCallMessage) message);
      } catch (ClassCastException cce) {
        throw new RPCIOException("Trying to write a RPC Message of type "+
                                 "call, object does not seem to be a "+
                                 "RPCCallMessage object. A cast exception "+
                                 "occured.");
      }
    }

    else if (messagetype == RPCMessage.MSGTYPE_REPLY) {
      try {
        writeReplyMessage((RPCReplyMessage) message);
      } catch (ClassCastException cce) {
        throw new RPCIOException("Trying to write a RPC Message of type "+
                                 "reply, object does not seem to be a "+
                                 "RPCReplyMessage object. A cast exception "+
                                 "occured.");
      }

    }

    else
      throw new RPCIOException("Unkown message type "+messagetype);


  }

  public void writeCallMessage(RPCCallMessage cmessage) 
         throws RPCIOException,
                IOException
  {

    int framelength = 0;
    RPCAuthentication creds = null,
                      verifier = null;

    if(cmessage == null)
      throw new RPCIOException("Can not write a null message");

    if(cmessage.getMessageType() != RPCMessage.MSGTYPE_CALL) 
      throw new RPCIOException("Trying to write a RPC Call Message that "+
                               "is not of call type. Type is "+
                               cmessage.getMessageType()+".");

    creds = cmessage.getCredentails();
    if (creds == null) 
      throw new RPCIOException("Can not write a RPC Call Message with no "+
                               "credentials.");

    verifier = cmessage.getVerifier();
    if (verifier == null)
      throw new RPCIOException("Can not write a RPC Call Message with no "+
                               "verifier.");

    framelength = cmessage.getLength();
    framelength = framelength << 1;
    writeInt(framelength);
   
    if (cmessage.getTransactionId() <= 0)
      cmessage.setTransactionId(xidgen.newXid()); 
    writeInt(cmessage.getTransactionId());
    writeInt(cmessage.getMessageType());
    writeInt(2); //The RPC version is always 2 -PICKARD
    writeInt(cmessage.getProgramNumber());
    writeInt(cmessage.getProgramVersion());
    writeInt(cmessage.getProcedureNumber());
    writeInt(creds.getAuthFlavor());
    if (creds.getRawParams() != null) 
      write(creds.getRawParams());
    writeInt(verifier.getAuthFlavor());
    if (verifier.getRawParams() != null)
      write(creds.getRawParams());
    write(cmessage.getParamaters());
    flush();

  }

  public void writeReplyMessage(RPCReplyMessage rmessage) 
         throws RPCIOException,
                IOException
  {

    int replytype = 0;

    if(rmessage == null)
      throw new RPCIOException("Can not write a null message");
   
    replytype = rmessage.getReplyMessageType();

    if(replytype == RPCReplyMessage.RPLY_ACCEPTED) {
      try {
        writeAcceptedMessage((RPCAcceptedReplyMessage) rmessage);
      }
      catch (ClassCastException cce) {
        throw new RPCIOException("Tried to write a RPC Reply Message "+
                                 "of type accepted, object does not "+
                                 "seem to be a RPCAccpetedReplyMessage "+
                                 "object. A cast exception occured.");
      }
    }

    else if(replytype == RPCReplyMessage.RPLY_DENIED) {
      try {
        writeRejectedMessage((RPCRejectedReplyMessage) rmessage);
      }
      catch (ClassCastException cce) {
        throw new RPCIOException("Tried to write a RPC Reply Message "+
                                 "of type rejected, object does not "+
                                 "seem to be a RPCRejectedReplyMessage "+
                                 "object. A cast exception occured.");
      }
    }

    else
      throw new RPCIOException("Trying to write a RPC Reply Message "+
                               "of unknown type "+replytype+".");
 
        

  }

  private void writeAcceptedMessage(RPCAcceptedReplyMessage armessage) 
          throws RPCIOException,
                 IOException
  {

    //Keep this method private so I know the types are good -PICKARD

    int framelength = 0,
        accepttype = 0;
    RPCAuthentication verifier = null; 

    accepttype = armessage.getAcceptedReplyMessageType();

    if ((accepttype < 0) || (accepttype > 5))
      throw new RPCIOException("Tried to write RPC Accepted Reply Message "+
                               "of unknown accept type "+accepttype+".");

    framelength = armessage.getLength();
    framelength = framelength << 1;
    writeInt(framelength);

    verifier = armessage.getVerifier();

    if (armessage.getTransactionId() <= 0)
      armessage.setTransactionId(xidgen.newXid());
    writeInt(armessage.getTransactionId());
    writeInt(armessage.getMessageType());
    writeInt(armessage.getReplyMessageType());
    writeInt(verifier.getAuthFlavor());
    if(verifier.getRawParams() != null)
      write(verifier.getRawParams());

    if (accepttype == RPCAcceptedReplyMessage.ACCPT_SUCCESS) {

      writeInt(accepttype);
      write(armessage.getParamaters());
      flush();
    }

    else if (accepttype == RPCAcceptedReplyMessage.ACCPT_PROG_MISMATCH) {
      
      writeInt(accepttype);
      writeInt(armessage.getProgramLowVersion());
      writeInt(armessage.getProgramHighVersion());
      flush();

    }

    else {
      writeInt(accepttype);
      flush();
    }

  }

  private void writeRejectedMessage(RPCRejectedReplyMessage rrmessage) 
          throws RPCIOException,
                 IOException
  {

    int framelength = 0,
        rejectedstat = 0;

    rejectedstat = rrmessage.getRejectedReplyMessageType();

    if ((rejectedstat != RPCRejectedReplyMessage.RJT_RPC_MISMATCH) &&
        (rejectedstat != RPCRejectedReplyMessage.RJT_AUTH_ERROR)) 
      throw new RPCIOException("Tried to write a RPC Rejected Reply Message "+
                               "of unknown reject type "+rejectedstat+".");
  
    framelength = rrmessage.getLength();
    framelength = framelength << 1;

    writeInt(framelength);

    if (rrmessage.getTransactionId() <= 0)
      rrmessage.setTransactionId(xidgen.newXid());
    writeInt(rrmessage.getTransactionId());
    writeInt(rrmessage.getMessageType());
    writeInt(rrmessage.getReplyMessageType());
    //No verifier for the Rejected Reply
    writeInt(rejectedstat);
    
    if (rejectedstat == RPCRejectedReplyMessage.RJT_RPC_MISMATCH) { 
      writeInt(rrmessage.getLowRPCVersion());
      writeInt(rrmessage.getHighRPCVersion());
      flush();
    }
    else {
      writeInt(rrmessage.getAuthenticationFailCode());
      flush();
    }

  }


}
